#include <unistd.h>
#include <sys/io.h>
#include <stdio.h>

#include "i2c.h"

int i2cDelay = 0;
int IOBASE; // typically 0x378

int SDA = 1;
int SCL = 1;
int debug = 0;

static inline void setupDelay() {
	/* the inb() and outb() calls include sufficient delay */
	int i;
	for(i = 0; i < i2cDelay; i++); /* needed for some devices */
//	usleep(5);
//	sleep(1);
}

static inline void pulldownSDA() {
	DEBUG( printf("SDA = %s SCL = %s\n", SDA?" / ":"|  ", SCL?"  |":"|  "); SDA = 0; )
	outb(0xFF & ((inb(IOBASE) | 0x1)), IOBASE);
	setupDelay();
}
static inline void releaseSDA() {
	DEBUG( printf("SDA = %s SCL = %s\n", SDA?"  |":" \\ ", SCL?"  |":"|  "); SDA = 1; )
	outb(0xFF & ((inb(IOBASE) & ~0x1)), IOBASE);
	setupDelay();
}
static inline void pulldownSCL() {
	DEBUG( printf("SDA = %s SCL = %s\n", SDA?"  |":"|  ", SCL?" / ":"  |"); SCL = 0; )
	outb(0xFF & ((inb(IOBASE) | 0x2)), IOBASE);
	setupDelay();
}
static inline void releaseSCL() {
	DEBUG( printf("SDA = %s SCL = %s\n", SDA?"  |":"|  ", SCL?"  |":" \\ "); SCL = 1; )
	outb(0xFF & ((inb(IOBASE) & ~0x2)), IOBASE);
	setupDelay();
}

static inline int readSDA() {
	DEBUG( printf("read SDA = %d\n", (inb(IOBASE + 1) >> 3) & 0x1); )
	return (inb(IOBASE + 1) >> 3) & 0x1;
}
static inline int readSCL() {
	DEBUG( printf("read SCL = %d\n", (inb(IOBASE + 1) >> 4) & 0x1); )
	return (inb(IOBASE + 1) >> 4) & 0x1;
}

/* asserts i2c start condition on the bus */
void i2cStart() {
	DEBUG( printf("enter i2cStart()\n"); )
	/* first we ensure both lines high
	  (this is also a stop condition if the data is currently low, a useful side effect) */
	releaseSCL();
	releaseSDA();

	/* then we drive the data line low to signal the start condition */
	pulldownSDA();

	/* now we can pull down the clock ready to send data */
	pulldownSCL();
	DEBUG( printf("exit i2cStart()\n"); )
}

/* asserts i2c stop condition on the bus */
void i2cStop() {
	DEBUG( printf("enter i2cStop()\n"); )
	/* first we ensure clock is high and data is low */
	pulldownSDA();
	releaseSCL();

	/* then we drive data high to signal the stop condition */
	releaseSDA();
	DEBUG( printf("exit i2cStop()\n"); )
}

/*
 * Sends byte over the i2c bus
 * Will return 0 on sucessful (acked) transfer)
 * Non-zero otherwise
 */
int i2cSendByte(int byte) {
	int i, ret;

	DEBUG( printf("start i2cSendByte(0x%.2X)\n", byte & 0xFF); )

	/* the clock should be currently low */
	
	/* send byte MSB first */
	for(i = 7; i > -1; i--) {
		(byte & (1 << i))?(releaseSDA()):(pulldownSDA());

		/* clock */
		releaseSCL();
		pulldownSCL();
	}

	/* get ACK - data has to be released so slave can pull it down */
	releaseSDA();
	releaseSCL();
	ret = readSDA();
	pulldownSCL();

	DEBUG( printf("exit i2cSendByte(), ack = %d\n", ret); )

	return ret;
}

/*
 * Reads a byte off the i2c bus
 * Will return byte value on sucessful (acked) transfer
 * Negative if ack was requested and failed
 */
int i2cReadByte(int ackControl) {
	int i, val = 0, ack;

	DEBUG( printf("start i2cReadByte(%d)\n", ackControl); )

	/* allow slave to pull data line - clock should currently be low */
	releaseSDA();

	/* read byte MSB first */
	for(i = 7; i > -1; i--) {
		releaseSCL();
		val |= (readSDA() << i);
		pulldownSCL();
	}

	switch(ackControl) {
		case EXPECT_ACK:
			releaseSCL();
			ack = readSDA();
			pulldownSCL();
			if(ack) return -1;
			break;
		case SEND_ACK:
			pulldownSDA();
			releaseSCL();
			pulldownSCL();
			releaseSDA();
			break;
		case SEND_NACK:
			releaseSDA();
			releaseSCL();
			pulldownSCL();
			pulldownSDA();
			break;
		default:
			fprintf(stderr, "error: unknown ack control mode!\n");
			i2cStop();
			exit(-1);
	}

	DEBUG( printf("exit i2cReadByte(), byte = 0x%.2X\n", val & 0xFF); )

	return val;
}

/*
 * set busy-wait delay between i2c transitions
 */
void i2cSetDelay(int delay) {
	i2cDelay = delay;
}

int i2cInit(int baseAddress) {
	IOBASE = baseAddress;

	if(ioperm(IOBASE, 3, 1)) {
		perror("ioperm(1)");
		return -1;
	}

	/* power on, raise all data and control bits, not just C0 because others may need to steal extra power */
	outb(0xFC, IOBASE);					/* D register - lower two bits are for SDA and SCL pulldown (inverted) */
	outb(((inb(IOBASE + 2) & 0xF0) | 0x4), IOBASE+ 2);	/* C register - lower nibble, all but C2 is inverted */
	return 0;
}

int i2cCleanUp() {
	/* power off */
	outb(0, IOBASE);
	outb(((inb(IOBASE + 2) & 0xF0) | 0xB), IOBASE + 2);

	if(ioperm(IOBASE, 3, 0)) {
		perror("ioperm(0)");
		return -1;
	}
	return 0;
}
