#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

#define OSCCTL _BV(PB0)
#define OUTPUT _BV(PB1)
#define OVERFLOW _BV(PB2)
#define METER _BV(PB3)

#define TRIGGER 1800
#define MAXCOUNT 2400

unsigned char doSample;

void init() {
	cli();

	DDRB |= OVERFLOW | OUTPUT | METER;
	PORTB = 0;
	
	// timer generates interrupts at ~100 Hz
	TCCR0A = _BV(WGM01);		// CTC mode
	TCCR0B = _BV(CS02) | _BV(CS00);	// Prescale 256 
	OCR0A = 94;			// TOP = 94
	TIMSK0 = _BV(OCIE0A);		// Interrupt at TOP

	doSample = 0;

	sei();
}

ISR(TIM0_COMPA_vect) {
	doSample = 1;
}

unsigned int sample() {
	unsigned int count;

	// tristate/input control pin to allow oscillator to start
	DDRB &= ~OSCCTL;

	// wait for oscillation to build
	for(count = 0; count < MAXCOUNT; count++) {
		if(PINB & OSCCTL) break;
		_delay_us(1);
	}

	// indicate overflow if oscillator failed to start
	// and assert output if signal strong enough
	PORTB = (PORTB & ~OVERFLOW) | ((count == MAXCOUNT)?(OVERFLOW):(0));
	PORTB = (PORTB & ~OUTPUT) | ((count < TRIGGER)?(OUTPUT):(0));

	// drive control pin low to stop oscillation
	PORTB &= ~OSCCTL;
	DDRB |= OSCCTL;	

	// we leave the control line low to let oscillation die away

	// return the sampled signal strength;
	return (MAXCOUNT - count);
}

int main() {
	unsigned int i, signal;

	init();

	while(1) {
		if(doSample) {
			// sample signal level
			signal = sample();

			// drive the RSSI meter pin
			PORTB |= METER;
			for(i = 0; i < signal; i++) _delay_us(1);
			PORTB &= ~METER;

			doSample = 0;
		}
		set_sleep_mode(SLEEP_MODE_IDLE);
		sleep_mode();
	}

	return 0; // unreached
}
