#include <stdlib.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <util/delay.h>
#include <avr/interrupt.h>

#include "instruction-set.h"
#include "program.h"

#define LINES 4
#define PINMASK 0x0F

#define STACK_DEPTH 3

unsigned char state[LINES];
unsigned char rampFadeMask;

unsigned char interrupts;
unsigned char interruptsPerInstruction;
unsigned char brightnessChangeRate;

unsigned char instruction;
unsigned char stack[STACK_DEPTH];
unsigned char counter[STACK_DEPTH];
unsigned char stackPointer;

void executeProgram() {
	unsigned char i;
	unsigned char bytecode = pgm_read_byte(&program[instruction++]);
	switch(bytecode & 0xF) {
		case WAIT:
			if(rampFadeMask) instruction --;	// WAIT pauses instruction processing until ramp/fade is complete by re-executing the WAIT.
			/* fall thru */
		case NOP:
		default:
			interrupts = 0;				// Relinquish CPU to PWM loop until next tick.
			return;					// All other instructions execute next instruction immediately.
		case ASSIGN:
		case TOGGLE:
			for(i = 0; i < LINES; i++) {
				switch(bytecode & 0xF) {
					case ASSIGN:
						state[i] = ((bytecode & (1 << (4 + i)))?(0xFF):(0));
						break;
					case TOGGLE:
						state[i] = ((bytecode & (1 << (4 + i)))?((state[i])?(0):(0xFF)):(state[i]));
						break;
				}
			}
			return;
		case RAMP:
			rampFadeMask = (rampFadeMask & 0x0F) | (bytecode & 0xF0);
			return;
		case FADE:
			rampFadeMask = (rampFadeMask & 0xF0) | ((bytecode >> 4) & 0x0F);
			return;
		case RATE:
			brightnessChangeRate = pgm_read_byte(&program[instruction++]);
			return;
		case SPEED:
			interruptsPerInstruction = pgm_read_byte(&program[instruction++]);
			return;
		case BEGIN:
			if(!(bytecode & 0x10)) {
				// FOR (otherwise we are executing a DO which has no counter)
				counter[stackPointer] = pgm_read_byte(&program[instruction++]);
			}
			stack[stackPointer++] = instruction;
			return;
		case END:
			// WHILE or NEXT
			if(((bytecode & 0x10) && rampFadeMask) || (!(bytecode & 0x10) && (--counter[stackPointer - 1]))) {
				instruction = stack[stackPointer - 1];
				return;
			}
			stackPointer --;
			return;
		case RESET:
			instruction = 0;
			stackPointer = 0;
			return;
	}
}

void rampFade() {
	unsigned char line;

	for(line = 0; line < LINES; line++) {
		if(rampFadeMask & (1 << (4 + line))) {
			if(state[line] > (0xFF - brightnessChangeRate)) {
				rampFadeMask &= ~(1 << (4 + line));
				state[line] = 0xFF;
			} else {
				state[line] += brightnessChangeRate;
			}
		}
		if(rampFadeMask & (1 << line)) {
			if(state[line] < brightnessChangeRate) {
				rampFadeMask &= ~(1 << line);
				state[line] = 0;
			} else {
				state[line] -= brightnessChangeRate;
			}
		}
	}
}

void pwm() {
	unsigned int i;
	unsigned char line;

	for(line = 0; line < LINES; line++) {
		if(state[line] > 0) PORTB |= (1 << line);;
	}

	for(i = 0; i < 0x100; i++) {
		for(line = 0; line < LINES; line++) {
			PORTB &= ((i > state[line])?(~(1 << line)):(PINMASK));
		}
	}

	PORTB = 0;
}

void init() {
/* 	//save a little space
	instruction = 0;
	stackPointer = 0;

	interrupts = 0;
	interruptsPerInstruction = 16;
	brightnessChangeRate = 1;
*/

	PORTB = 0;			// start with everything off
	DDRB = PINMASK;

	TCCR0A = _BV(WGM01);		// Clear timer on compare match mode
	TCCR0B = _BV(CS02) | _BV(CS00);	// Clock prescale by 1024
	OCR0A = 0xff;			// Compare match at top
	TIMSK0 = _BV(OCIE0A);		// Enable compare match interrupts

	sei();
}

ISR(TIM0_COMPA_vect) {
	if(rampFadeMask) rampFade();
	interrupts++;
}

int main() {

	init();
	while(1) {
		while(interrupts >= interruptsPerInstruction) executeProgram();
		pwm();
	}

	return 0; // unreached
}
